#pragma once
#include <unordered_set>
#include <boost/range/algorithm.hpp>
#include <boost/checked_delete.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <boost/cast.hpp>

namespace Jenga{
namespace Common{

template<class T>
class ObjectInHashmap;
template<class T>
struct ObjectInHashmapHash;
template<class T>
struct ObjectInHashmapEqualTo;

#define MAX_HASHMAP 65535
template<class T> class Hashmap : boost::noncopyable
{
	typedef std::unordered_set<ObjectInHashmap<T>*, ObjectInHashmapHash<T>, ObjectInHashmapEqualTo<T>> MapType;
	MapType map;

	struct downcast
	{
		typedef T* result_type;
		T* operator ()(ObjectInHashmap<T>* p) const
		{
			return boost::polymorphic_cast<T*>(p);
		}
	};
	struct const_downcast
	{
		typedef T const* result_type;
		T const* operator ()(ObjectInHashmap<T> const* p) const
		{
			return boost::polymorphic_cast<T const*>(p);
		}
	};

public:

	Hashmap()
	{
	}
	~Hashmap()
	{
		Clear();
	}
	void Clear()
	{
		boost::for_each(*this, boost::checked_deleter<T const>());
		map.clear();
	}

	// ejɂׂĔ
	void PullOutAll()
	{
		map.clear();
	}

	bool Put(T* value)
	{
		if (value == nullptr)
		{
			throw std::invalid_argument("Hashmap::Put");
		}
		return map.insert(value).second;
	}

	typedef boost::transform_iterator<downcast, typename MapType::local_iterator> local_iterator;
	typedef boost::transform_iterator<const_downcast, typename MapType::const_local_iterator> const_local_iterator;

	boost::iterator_range<local_iterator> GetHashArrayElement(std::string const& keyName)
	{
		ObjectInHashmapDummy<T> t(keyName);
		return boost::iterator_range<local_iterator>(
			local_iterator(map.begin(map.bucket(&t)), downcast()),
			local_iterator(map.end(map.bucket(&t)), downcast()));
	}

	boost::iterator_range<const_local_iterator> GetHashArrayElement(std::string const& keyName) const
	{
		ObjectInHashmapDummy<T> t(keyName);
		return boost::iterator_range<const_local_iterator>(
			const_local_iterator(map.begin(map.bucket(&t)), const_downcast()),
			const_local_iterator(map.end(map.bucket(&t)), const_downcast()));
	}

	bool IsExistDuplicationKeyName(const std::string &keyName) const
	{
		ObjectInHashmapDummy<T> t(keyName);
		return map.find(&t) != map.end();
	}

	bool IsExist( const T* value ) const
	{
		return map.find(const_cast<T*>(value)) != map.end();
	}

	const T *FindLike( const ObjectInHashmap<T>* value ) const
	{
		auto it = map.find(const_cast<ObjectInHashmap<T>*>(value));
		return it != map.end()
			? boost::polymorphic_downcast<T const*>(*it)
			: nullptr;
	}


	/////////////////////////////////////////////////////////////////
	// Ce[^
	/////////////////////////////////////////////////////////////////
	//typedef boost::transform_iterator<downcast, typename MapType::iterator> iterator;
	typedef boost::transform_iterator<downcast, typename MapType::const_iterator> const_iterator;
	typedef const_iterator iterator;
	typedef typename MapType::size_type size_type;
	typedef typename MapType::difference_type difference_type;
	//iterator begin()
	//{
	//	return iterator(map.begin(), downcast());
	//}
	//iterator end()
	//{
	//	return iterator(map.end(), downcast());
	//}
	const_iterator begin() const
	{
		return const_iterator(map.begin(), downcast());
	}
	const_iterator end() const
	{
		return const_iterator(map.end(), downcast());
	}

	// XMLVACYp
private:
	friend class boost::serialization::access;
	BOOST_SERIALIZATION_SPLIT_MEMBER();
	template<class Archive> void load(Archive& ar, const unsigned int version)
	{
		std::vector<T *> objects;
		ar & BOOST_SERIALIZATION_NVP(objects);
		Clear();
		map = boost::copy_range<MapType>(objects);
	}
	template<class Archive> void save(Archive& ar, const unsigned int version) const
	{
		std::vector<T *> objects(begin(), end());
		ar & BOOST_SERIALIZATION_NVP(objects);
	}
};

template<class T>
class ObjectInHashmap
{
protected:
	ObjectInHashmap()
	{
	}
	~ObjectInHashmap()
	{
	}

public:
	virtual const std::string &GetKeyName() const = 0;
	virtual bool IsDuplication( const T *value ) const
	{
		return ( GetKeyName() == value->GetName() );
	}
	virtual bool IsDuplication( const std::string &keyName ) const
	{
		return ( GetKeyName() == keyName );
	}
};

template<class T>
struct ObjectInHashmapHash
{
	typedef std::size_t result_type;
	std::size_t operator ()(ObjectInHashmap<T> const* x) const
	{
		return std::hash<std::string>()(x->GetKeyName());
	}
};

// GetHashArrayElementȂǂŕ񂩂猟邽߂ɗpB
template<class T>
class ObjectInHashmapDummy : public ObjectInHashmap<T>, boost::noncopyable
{
public:
	explicit ObjectInHashmapDummy(std::string const& s) : str(s) {}

	virtual std::string const& GetKeyName() const override
	{
		return str;
	}

	virtual bool IsDuplication(T const* value) const override
	{
		return value != nullptr
			&& value->ObjectInHashmap<T>::IsDuplication(str);
	}

private:
	std::string const& str;
};

template<class T>
struct ObjectInHashmapEqualTo
{
	typedef bool result_type;
	bool operator ()(_In_ ObjectInHashmap<T> const* lhs, _In_ ObjectInHashmap<T> const* rhs) const
	{
		assert(lhs != nullptr);
		assert(rhs != nullptr);
		if (auto pl = dynamic_cast<T const*>(rhs))
		{
			return lhs->IsDuplication(pl);
		}
		else
		{
			return rhs->IsDuplication(dynamic_cast<T const*>(lhs));
		}
	}
};


}}
