#include "stdafx.h"
#include <vector>
#include <utility>
#include <boost/range.hpp>
#include <boost/range/adaptor/transformed.hpp>
#include <boost/numeric/conversion/cast.hpp>
#include <Resource/Load.h>

using boost::numeric_cast;

namespace ActiveBasic { namespace Resource {

namespace {

template<typename Map, typename Key, typename AddValueFunctor>
typename Map::iterator GetCacheFromMap(Map& map, Key key, AddValueFunctor f)
{
	auto low = map.lower_bound(key);
	if (low != map.end() && map.key_comp()(key, low->first))
	{
		return low;
	}
	else
	{
		typedef typename Map::key_type key_type;
		typedef typename Map::value_type value_type;
		return map.insert(low, std::make_pair(std::move(key), f()));
	}
}

void* GetResource(HINSTANCE hinst, USHORT id, LPCTSTR type)
{
	if (auto hrsrc = ::FindResource(hinst, MAKEINTRESOURCE(id), type))
	{
		if (auto hSrc = ::LoadResource(hinst, hrsrc))
		{
			return ::LockResource(hSrc);
		}
	}
	return nullptr;
}

std::pair<void*, DWORD> GetResourceWithSize(HINSTANCE hinst, USHORT id, LPCTSTR type)
{
	typedef std::pair<void*, DWORD> result_type;
	if (auto hrsrc = ::FindResource(hinst, MAKEINTRESOURCE(id), type))
	{
		if (auto hSrc = ::LoadResource(hinst, hrsrc))
		{
			auto p = ::LockResource(hSrc);
			auto size = ::SizeofResource(hinst, hrsrc);
			if (p != nullptr && size > 0)
			{
				return result_type(p, size);
			}
		}
	}
	return result_type(nullptr, 0);
}

HICON LoadIconCursorImpl(HINSTANCE hinst, USHORT id, int cxDesired, int cyDesired, UINT load, bool isIcon)
{
	auto pResource = GetResource(hinst, id, isIcon ? RT_GROUP_ICON : RT_GROUP_CURSOR);

	auto idIcon = LookupIconIdFromDirectoryEx(reinterpret_cast<PBYTE>(pResource), isIcon, cxDesired, cyDesired, load);
	auto icon = GetResourceWithSize(hinst, numeric_cast<USHORT>(idIcon), isIcon ? RT_ICON : RT_CURSOR);

	return CreateIconFromResourceEx(reinterpret_cast<PBYTE>(icon.first),
		icon.second, isIcon, 0x00030000, cxDesired, cyDesired, load);
}

} // unnamed namespace

HICON LoadIconAlt(HINSTANCE hinst, USHORT id, int cxDesired, int cyDesired, UINT load)
{
	return LoadIconCursorImpl(hinst, id, cxDesired, cyDesired, load, true);
}

HICON LoadIconAlt(HINSTANCE hinst, USHORT id)
{
	return LoadIconCursorImpl(hinst, id, 32, 32, LR_SHARED, true);
}

HCURSOR LoadCursorAlt(HINSTANCE hinst, USHORT id)
{
	return LoadIconCursorImpl(hinst, id, 32, 32, LR_SHARED, false);
}

// 񃊃\[X̓ǂݍ݂ɂĂ͈ȉQƁB
// How To Use LoadResource to Load Strings from a String Table
// http://support.microsoft.com/kb/200893/en-us
boost::optional<std::wstring> LoadStringAlt(HINSTANCE hinst, USHORT id)
{
	UINT idRsrcBlk = id / 16 + 1;
	int strIndex  = id % 16;

	auto hrsrc = FindResourceEx(hinst, RT_STRING, MAKEINTRESOURCE(idRsrcBlk), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT));
	if (hrsrc == nullptr)
		return boost::none;

	auto hRes = LoadResource(hinst, hrsrc);
	if (hRes == nullptr)
		return boost::none;
	LPCWSTR p = static_cast<LPCWSTR>(LockResource(hRes));
	if (p == nullptr)
		return boost::none;

	for (int i = 0; i < strIndex; ++i)
	{
		UINT length = *p++;
		p += length;
	}

	UINT cch = *p++;
	if (cch == 0)
	{
		return boost::none;
	}
	else
	{
		return std::wstring(p, cch);
	}
}

namespace {

// http://msdn.microsoft.com/en-us/library/ms648010.aspx
struct ACCELTABLEENTRY
{
	WORD fFlags;
	WORD wAnsi;
	WORD wId;
	WORD padding;
};

ACCEL AccelFromEntry(ACCELTABLEENTRY const& e)
{
	ACCEL t = {static_cast<BYTE>(e.fFlags & 0x7f), e.wAnsi, e.wId};
	return t;
}

} // unnamed namespace

HACCEL LoadAcceleratorsAlt(HINSTANCE hinst, USHORT id)
{
	using namespace boost::adaptors;

	auto accel = GetResourceWithSize(hinst, id, RT_ACCELERATOR);
	if (accel.first == nullptr)
	{
		return nullptr;
	}

	auto pate = static_cast<ACCELTABLEENTRY const*>(accel.first);

	auto a = boost::copy_range<std::vector<ACCEL>>(
		boost::make_iterator_range(pate, pate + accel.second / sizeof (ACCELTABLEENTRY))
		| transformed(AccelFromEntry));

	if (a.size() > INT_MAX)
	{
		a.resize(INT_MAX);
	}
	return CreateAcceleratorTable(&a[0], static_cast<int>(a.size()));
}

HMENU LoadMenuAlt(HINSTANCE hinst, USHORT id)
{
	return LoadMenuIndirect(GetResource(hinst, id, RT_MENU));
}

INT_PTR DialogBoxAlt(HINSTANCE hinst, USHORT id, HWND hwndParent, DLGPROC dialogFunc, LPARAM initParam)
{
	auto dlgTemplate = static_cast<DLGTEMPLATE*>(GetResource(hinst, id, RT_DIALOG));
	return ::DialogBoxIndirectParam(hinst, dlgTemplate, hwndParent, dialogFunc, initParam);
}

HWND CreateDialogAlt(HINSTANCE hinst, USHORT id, HWND hwndParent, DLGPROC dialogFunc, LPARAM initParam)
{
	auto dlgTemplate = static_cast<DLGTEMPLATE*>(GetResource(hinst, id, RT_DIALOG));
	return ::CreateDialogIndirectParam(hinst, dlgTemplate, hwndParent, dialogFunc, initParam);
}

}}
