#pragma once

#include <vector>

#include <jenga/include/common/Exception.h>

#include <BoostSerializationSupport.h>

class UserProc;

class Schedule
{
public:
	enum Type
	{
		None = 10000,
		GlobalVar,		// O[oϐXPW[
		DataTable,		// f[^e[u XPW[
		Relocation,		// P[VXPW[
		UserProc,		// [U`֐ĂяoXPW[
		AddressOf,		// [U`֐ʒuXPW[
		DllProc,		// DLL֐ʒuXPW[
	};

private:
	Type type;
	long offset;

	union{
		LONG_PTR lpValue;
		const ::UserProc *pUserProc;
		const ::DllProc *pDllProc;
	};

	// XMLVACYp
private:
	friend class boost::serialization::access;
	template<class Archive> void serialize(Archive& ar, const unsigned int version)
	{
		trace_for_serialize( "serializing - Schedule" );

		ar & BOOST_SERIALIZATION_NVP( type );
		ar & BOOST_SERIALIZATION_NVP( offset );

		switch( type )
		{
		case UserProc:
		case AddressOf:
			ar & boost::serialization::make_nvp("pUserProc", const_cast<::UserProc *&>(pUserProc));
			break;
		case DllProc:
			ar & boost::serialization::make_nvp("pDllProc", const_cast<::DllProc *&>(pDllProc));
			break;
		default:
			ar & BOOST_SERIALIZATION_NVP( lpValue );
			break;
		}
	}

public:
	Schedule()
	{
	}
	Schedule( Type type, long offset, LONG_PTR lpValue = 0 )
		: type( type )
		, offset( offset )
		, lpValue( lpValue )
	{
	}
	Schedule( const ::UserProc *pUserProc, long offset )
		: type( Schedule::UserProc )
		, offset( offset )
		, pUserProc( pUserProc )
	{
	}
	Schedule( const ::DllProc *pDllProc, long offset )
		: type( Schedule::DllProc )
		, offset( offset )
		, pDllProc( pDllProc )
	{
	}
	~Schedule()
	{
	}

	Type GetType() const
	{
		return type;
	}
	long GetOffset() const
	{
		return offset;
	}
	LONG_PTR GetLongPtrValue() const
	{
		return lpValue;
	}
	const ::DllProc &GetDllProc() const
	{
		if( type != Schedule::DllProc )
		{
			SetError();
		}
		return *pDllProc;
	}
	const ::UserProc &GetUserProc() const
	{
		if( !( type == Schedule::UserProc || type == Schedule::AddressOf ) )
		{
			SetError();
		}
		return *pUserProc;
	}

	void SpecifyAddressOf()
	{
		if( type != Schedule::UserProc )
		{
			SetError();
		}
		type = Schedule::AddressOf;
	}
};
typedef std::vector<Schedule> Schedules;

#define CODETYPE_SYSTEMPROC		0x0001
#define CODETYPE_DEBUGPROC		0x0002
class SourceLine
{
	int lineNum;
	long nativeCodePos;
	int sourceIndex;
	long sourceCodePos;
	DWORD codeType;

	// XMLVACYp
private:
	friend class boost::serialization::access;
	template<class Archive> void serialize(Archive& ar, const unsigned int version)
	{
		trace_for_serialize( "serializing - SourceLine" );

		ar & BOOST_SERIALIZATION_NVP( lineNum );
		ar & BOOST_SERIALIZATION_NVP( nativeCodePos );
		ar & BOOST_SERIALIZATION_NVP( sourceIndex );
		ar & BOOST_SERIALIZATION_NVP( sourceCodePos );
		ar & BOOST_SERIALIZATION_NVP( codeType );
	}

public:
	SourceLine( int lineNum, int nativeCodePos, int sourceIndex, int sourceCodePos, DWORD codeType )
		: lineNum( lineNum )
		, nativeCodePos( nativeCodePos )
		, sourceIndex( sourceIndex )
		, sourceCodePos( sourceCodePos )
		, codeType( codeType )
	{
	}
	SourceLine()
	{
	}

	int GetLineNum() const
	{
		return lineNum;
	}
	long GetNativeCodePos() const
	{
		return nativeCodePos;
	}
	int GetSourceIndex() const
	{
		return sourceIndex;
	}
	void SetSourceIndex( int sourceIndex )
	{
		this->sourceIndex = sourceIndex;
	}
	long GetSourceCodePos() const
	{
		return sourceCodePos;
	}
	void SetSourceCodePos( int sourceCodePos )
	{
		this->sourceCodePos = sourceCodePos;
	}
	DWORD GetCodeType() const
	{
		return codeType;
	}
	bool IsInSystemProc() const
	{
		return ( (codeType&CODETYPE_SYSTEMPROC) != 0 );
	}
	bool IsInDebugProc() const
	{
		return ( (codeType&CODETYPE_DEBUGPROC) != 0 );
	}
};
typedef std::vector<SourceLine> SourceLines;

class NativeCode
{
	int allocateSize;
	char *codeBuffer;
	int size;

	// JŉȂ΂ȂȂXPW[
	Schedules schedules;

	// \[XR[hsԍƃlCeBuR[hʒȗΉ
	SourceLines sourceLines;

	// XMLVACYp
private:
	friend class boost::serialization::access;
	BOOST_SERIALIZATION_SPLIT_MEMBER();
	template<class Archive> void load(Archive& ar, const unsigned int version)
	{
		trace_for_serialize( "serializing(load) - NativeCode" );

		std::string code;
		ar & BOOST_SERIALIZATION_NVP( code );
		ar & BOOST_SERIALIZATION_NVP( size );
		ar & BOOST_SERIALIZATION_NVP( schedules );
		ar & BOOST_SERIALIZATION_NVP( sourceLines );

		// ǂݍ݌̏
		Realloc( size );
		for( int i=0; i<size; i++ )
		{
			ULONG_PTR l1 = ( ( code[i*3] >= 'a' ) ? ( code[i*3] - 'a' + 0x0a ) : ( code[i*3] - '0' ) ) * 0x10;
			ULONG_PTR l2 = ( code[i*3+1] >= 'a' ) ? ( code[i*3+1] - 'a' + 0x0a ) : ( code[i*3+1] - '0' );
			ULONG_PTR l = l1 + l2;
			codeBuffer[i] = static_cast<char>(l);
		}
	}
	template<class Archive> void save(Archive& ar, const unsigned int version) const
	{
		trace_for_serialize( "serializing(save) - NativeCode" );

		// ۑ
		char *tempCode = (char *)calloc( (size+1) * 3, 1 );
		for( int i=0; i<size; i++ )
		{
			char temp[32];
			sprintf( temp, "%02x,", (unsigned char)codeBuffer[i] );
			tempCode[i*3] = temp[0];
			tempCode[i*3+1] = temp[1];
			tempCode[i*3+2] = temp[2];
		}

		std::string code = tempCode;
		free( tempCode );

		ar & BOOST_SERIALIZATION_NVP( code );
		ar & BOOST_SERIALIZATION_NVP( size );
		ar & BOOST_SERIALIZATION_NVP( schedules );
		ar & BOOST_SERIALIZATION_NVP( sourceLines );
	}


	void Realloc( int newSize )
	{
		if( allocateSize < newSize + 8192 )
		{
			while( allocateSize < newSize + 8192 )
			{
				allocateSize += 8192;
			}
			codeBuffer = (char *)realloc( codeBuffer, allocateSize );
		}
	}

public:
	NativeCode()
		: allocateSize( 8192 )
		, codeBuffer( (char *)malloc( allocateSize ) )
		, size( 0 )
	{
	}
	NativeCode( const NativeCode &nativeCode )
		: allocateSize( 8192 )
		, codeBuffer( (char *)malloc( allocateSize ) )
		, size( 0 )
	{
		Put( nativeCode );
	}
	NativeCode( const char *codeBuffer, int size )
		: allocateSize( 8192 )
		, codeBuffer( (char *)malloc( allocateSize ) )
		, size( 0 )
	{
		Put( codeBuffer, size );
	}
	~NativeCode()
	{
		free( codeBuffer );
	}
	void Clear()
	{
		size = 0;
	}

	void operator =( const NativeCode &nativeCode )
	{
		Clear();
		Put( nativeCode );
	}

	const char *GetCodeBuffer() const
	{
		return codeBuffer;
	}
	int GetSize() const
	{
		return size;
	}
	const Schedules &GetSchedules() const
	{
		return schedules;
	}

	long GetLong( int codePos ) const
	{
		return *(long *)(this->codeBuffer+codePos);
	}

	void Overwrite( int codePos, char c )
	{
		codeBuffer[codePos] = c;
	}
	void Overwrite( int codePos, long newLongValue )
	{
		*(long *)(this->codeBuffer+codePos) = newLongValue;
	}

	void Put( const char *codeBuffer, int size )
	{
		Realloc( this->size + size );

		memcpy( this->codeBuffer + this->size, codeBuffer, size );
		this->size += size;
	}
	void Put( const NativeCode &nativeCode );
	void Put( _int64 i64data )
	{
		Put( (const char *)(&i64data), sizeof(_int64) );
	}
	void Put( long l, Schedule::Type scheduleType = Schedule::None )
	{
		if( scheduleType != Schedule::None )
		{
			schedules.push_back( Schedule( scheduleType, size ) );
		}

		*((long *)(codeBuffer+size))=l;
		size += sizeof(long);
	}
	void PutUserProcSchedule( const UserProc *pUserProc, bool isCall );
	void PutDllProcSchedule( const DllProc *pDllProc );
	void Put( short s )
	{
		Put( (const char *)(&s), sizeof(short) );
	}
	void Put( char c )
	{
		Realloc( size + 1 );
		codeBuffer[size++] = c;
	}

	const SourceLines &GetSourceLines() const
	{
		return sourceLines;
	}
	void NextSourceLine();

	void ResetDataSectionBaseOffset( long dataSectionBaseOffset );
	void ResetSourceIndexes( long sourceIndexBase );
};
