#pragma once

#include <string>
#include <vector>

#include <boost/foreach.hpp>

#include <jenga/include/common/Exception.h>
#include <BoostSerializationSupport.h>
#include <jenga/include/smoothie/BasicFixed.h>

#include <option.h>
#include <Program.h>

#include <windows.h>

class CClass;

class Type{
	int basicType;
	union{
		LONG_PTR index;
		const CClass *pClass;
	};

	// XMLVACYp
private:
	friend class boost::serialization::access;
	template<class Archive> void serialize(Archive& ar, const unsigned int version)
	{
		trace_for_serialize( "serializing - Type" );

		ar & BOOST_SERIALIZATION_NVP( basicType );
		if( HasMember() )
		{
			ar & boost::serialization::make_nvp("pClass", const_cast<CClass *&>(pClass));
		}
		else
		{
			ar & BOOST_SERIALIZATION_NVP( index );
		}
	}

public:
	static int GetBasicSize( int basicType );

	Type():
	  basicType( DEF_NON ),
	  index( -1 ){}
	Type( int basicType ):
	  basicType( basicType ),
	  index( -1 ){}

	Type( int basicType, LONG_PTR index ):
	  basicType( basicType ),
	  index( index ){}

	Type( int basicType, const CClass &objClass ):
	  basicType( basicType ),
	  index( (LONG_PTR)&objClass ){}

	Type( const Type &type ):
	  basicType( type.basicType ),
	  index( type.index ){}

	__inline int GetBasicType() const
	{
		return basicType;
	}
	LONG_PTR GetIndex() const
	{
		return index;
	}
	const CClass &GetClass() const
	{
		return *pClass;
	}

	void SetBasicType( int basicType ){
		this->basicType = basicType;
	}
	void SetIndex( LONG_PTR index ){
		this->index = index;
	}
	void SetClassPtr( const CClass *pClass )
	{
		this->pClass = pClass;
	}
	void SetNull(){
		SetBasicType( DEF_NON );
		SetIndex( -1 );
	}
	void SetType( int basicType, LONG_PTR index ){
		SetBasicType( basicType );
		SetIndex( index );
	}
	void SetType( int basicType, const CClass *pClass ){
		SetBasicType( basicType );
		this->pClass = pClass;
	}

	int PtrLevel() const
	{
		return PTR_LEVEL( basicType );
	}
	void PtrLevelUp(){
		PTR_LEVEL_UP( basicType );
	}
	void PtrLevelDown(){
		PTR_LEVEL_DOWN( basicType );
	}

	bool Equals( const Type &type ) const;

	int GetBasicSize() const;
	int GetSize() const;

	bool IsNull() const;

	bool IsByte() const;
	bool IsSByte() const;
	bool IsWord() const;
	bool IsInteger() const;
	bool IsDWord() const;
	bool IsLong() const;
	bool IsQWord() const;
	bool IsInt64() const;
	bool IsSingle() const;
	bool IsDouble() const;
	bool IsBoolean() const;

	static bool IsPointer( int basicType );
	bool IsPointer() const;
	bool IsSigned() const;
	bool IsNaturalWhole() const;
	bool IsWhole() const;
	bool IsReal() const;
	bool Is64() const;
	bool IsProcPtr() const;
	bool IsStruct() const;
	bool IsStructPtr() const;
	bool IsObject() const;
	bool IsObjectPtr() const;
	bool IsObjectClass() const;
	bool IsStringClass() const;
	bool IsVoidPtr() const;
	bool IsAny() const;

	// IuWFNg\̂ȂǁAo^ǂ𔻕ʂ
	bool HasMember() const;

	void operator= ( const Type &type ){
		basicType = type.basicType;
		index = type.index;
	}


private:
	static const int basicTypeList[];
	static const std::string basicTypeNameList[];
public:
	static bool StringToBasicType( const std::string &typeName, int &basicType );
	static const char *Type::BasicTypeToCharPtr( const Type &type );
	static int GetBasicTypeFromSimpleName( const char *variable );
};

class BlittableType
{
	Type basicType;
	CClass *pClass;

	// XMLVACYp
private:
	friend class boost::serialization::access;
	template<class Archive> void serialize(Archive& ar, const unsigned int version)
	{
		trace_for_serialize( "serializing - BlittableType" );

		ar & BOOST_SERIALIZATION_NVP( basicType );
		ar & BOOST_SERIALIZATION_NVP( pClass );
	}

public:
	bool isTargetObjectModule;
	BlittableType( const Type &basicType, CClass *pClass )
		: basicType( basicType )
		, pClass( pClass )
		, isTargetObjectModule( true )
	{
	}
	BlittableType()
		: isTargetObjectModule( true )
	{
	}
	const Type &GetBasicType() const
	{
		return basicType;
	}
	const CClass *GetClassPtr() const
	{
		return pClass;
	}
	const std::string GetCreateStaticMethodFullName() const;
};
class BlittableTypes : public std::vector<BlittableType>
{
	// XMLVACYp
private:
	friend class boost::serialization::access;
	template<class Archive> void serialize(Archive& ar, const unsigned int version)
	{
		trace_for_serialize( "serializing - BlittableTypes" );

		ar & boost::serialization::make_nvp("vector_BlittableType",
			boost::serialization::base_object<std::vector<BlittableType>>(*this));
	}

public:
	bool IsExist( Type type ) const
	{
		const BlittableTypes &blittableTypes = *this;
		BOOST_FOREACH( const BlittableType &blittableType, blittableTypes ){
			if( blittableType.GetBasicType().Equals( type ) ){
				return true;
			}
		}
		return false;
	}
	const BlittableType &Find( const Type &type ) const
	{
		const BlittableTypes &blittableTypes = *this;
		BOOST_FOREACH( const BlittableType &blittableType, blittableTypes ){
			if( blittableType.GetBasicType().Equals( type ) ){
				return blittableType;
			}
		}
		Jenga::Throw( "Blittable^ł͂Ȃ" );

		static BlittableType dummy;
		return dummy;
	}
	const CClass *GetClassPtr( const Type &type ) const
	{
		const BlittableTypes &blittableTypes = *this;
		BOOST_FOREACH( const BlittableType &blittableType, blittableTypes ){
			if( blittableType.GetBasicType().Equals( type ) ){
				return blittableType.GetClassPtr();
			}
		}
		return NULL;
	}
	const CClass &GetClass( const Type &type ) const
	{
		return *GetClassPtr( type );
	}
};
